const state = {
  settings: {},
  catalog: {categories:[], subcategories:[], brands:[]},
  products: [],
  cart: {},
  activeCat: '',
  activeSub: ''
};

async function api(path){
  const url = path + (path.includes('?') ? '&' : '?') + 't=' + Date.now();
  const res = await fetch(url, {cache:'no-store'});
  if(!res.ok) throw new Error('API error');
  return res.json();
}

function esc(s){return String(s??'').replace(/[&<>"']/g, m=>({ '&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;','\'':'&#39;' }[m]));}

function setHeader(){
  const nameEl = document.getElementById('siteNameText');
  const logoEl = document.getElementById('siteLogo');
  if(nameEl) nameEl.textContent = state.settings.name || 'eNeed.shop';
  if(logoEl){
    if(state.settings.logoUrl){
      logoEl.src = state.settings.logoUrl;
      logoEl.classList.remove('hidden');
    } else {
      logoEl.classList.add('hidden');
    }
  }
}

function renderNavCategories(){
  const wrap = document.getElementById('navCatDropdown');
  if(!wrap) return;
  const cats = state.catalog.categories;
  const subs = state.catalog.subcategories;
  const byCat = {};
  subs.forEach(s=>{(byCat[s.categoryId] ||= []).push(s);});

  if(!cats.length){
    wrap.innerHTML = '<div class="px-4 py-2 text-xs text-gray-400 font-bold">No categories</div>';
    return;
  }

  wrap.innerHTML = cats.map(c=>{
    const list = byCat[c.id] || [];
    if(!list.length){
      return `<a href="#" class="block px-4 py-2 hover:bg-blue-50 font-bold" data-cat="${esc(c.id)}">${esc(c.name)}</a>`;
    }
    const subHtml = list.map(s=>`<a href="#" class="block px-4 py-2 hover:bg-blue-50 font-bold" data-sub="${esc(s.id)}" data-sub-cat="${esc(c.id)}">${esc(s.name)}</a>`).join('');
    return `
      <div class="relative group">
        <a href="#" class="flex justify-between items-center px-4 py-2 hover:bg-blue-50 font-bold" data-cat="${esc(c.id)}">
          <span>${esc(c.name)}</span><span class="text-[10px] opacity-60">▶</span>
        </a>
        <div class="absolute left-full top-0 hidden group-hover:block bg-white shadow-xl border w-56 py-2 r5">
          ${subHtml}
        </div>
      </div>
    `;
  }).join('');

  wrap.querySelectorAll('[data-cat]').forEach(a=>a.addEventListener('click', (e)=>{
    e.preventDefault();
    selectCategory(a.getAttribute('data-cat'));
  }));
  wrap.querySelectorAll('[data-sub]').forEach(a=>a.addEventListener('click', (e)=>{
    e.preventDefault();
    selectCategory(a.getAttribute('data-sub-cat'));
    selectSubcategory(a.getAttribute('data-sub'));
  }));
}

function renderCatRows(){
  const top = document.getElementById('catMenuTop');
  const topSub = document.getElementById('subMenuTop');
  const bot = document.getElementById('catMenuBottom');
  const botSub = document.getElementById('subMenuBottom');

  const cats = state.catalog.categories;
  if(top) top.innerHTML = cats.map(c=>`<button data-catbtn="${esc(c.id)}">${esc(c.name)}</button>`).join('');
  if(bot) bot.innerHTML = cats.map(c=>`<button data-catbtn="${esc(c.id)}">${esc(c.name)}</button>`).join('');
  if(topSub) topSub.innerHTML = '';
  if(botSub) botSub.innerHTML = '';

  document.querySelectorAll('[data-catbtn]').forEach(b=>b.addEventListener('click', ()=>{
    selectCategory(b.getAttribute('data-catbtn'));
  }));
}

function renderSubRows(catId){
  const subs = state.catalog.subcategories.filter(s=>s.categoryId === catId);
  const html = subs.map(s=>`<button data-subbtn="${esc(s.id)}">${esc(s.name)}</button>`).join('');
  const topSub = document.getElementById('subMenuTop');
  const botSub = document.getElementById('subMenuBottom');
  if(topSub) topSub.innerHTML = html;
  if(botSub) botSub.innerHTML = html;
  document.querySelectorAll('[data-subbtn]').forEach(b=>b.addEventListener('click', ()=>{
    selectSubcategory(b.getAttribute('data-subbtn'));
  }));
}

function selectCategory(catId){
  state.activeCat = catId;
  state.activeSub = '';
  // active UI
  document.querySelectorAll('[data-catbtn]').forEach(b=>b.classList.toggle('active', b.getAttribute('data-catbtn')===catId));
  renderSubRows(catId);
  renderProducts();
    bindCartUI();
    syncCartCount();
    renderCart();
}

function selectSubcategory(subId){
  state.activeSub = subId;
  document.querySelectorAll('[data-subbtn]').forEach(b=>b.classList.toggle('active', b.getAttribute('data-subbtn')===subId));
  renderProducts();
    bindCartUI();
    syncCartCount();
    renderCart();
}

function renderSlider(){
  const wrap = document.getElementById('heroSliderWrap');
  if(!wrap) return;
  const slides = state.slider || [];
  if(!slides.length){ wrap.innerHTML = '<div class="p-10 text-center text-gray-400 font-black">No slider</div>'; return; }
  wrap.innerHTML = `
    <div class="relative">
      <img id="sliderImg" src="${esc(slides[0].url)}" class="w-full h-56 md:h-72 object-cover"/>
      <div class="absolute bottom-3 left-3 bg-white/80 r5 px-3 py-2 text-xs font-black">${esc(slides[0].title||'')}</div>
    </div>
  `;
  let i=0;
  setInterval(()=>{
    if(!slides.length) return;
    i = (i+1) % slides.length;
    const img = document.getElementById('sliderImg');
    if(img) img.src = slides[i].url;
  }, 3500);
}

function renderProducts(){
  const grid = document.getElementById('productGrid');
  if(!grid) return;
  let list = state.products.slice();
  if(state.activeCat) list = list.filter(p=>p.cat === state.activeCat);
  if(state.activeSub) list = list.filter(p=>p.sub === state.activeSub);
  grid.innerHTML = list.map(p=>`
    <div class="card p-2">
      <div class="bg-slate-50 r5 overflow-hidden">
        <img src="${esc(p.image || '')}" class="w-full h-32 md:h-40 object-contain"/>
      </div>
      <div class="mt-2">
        <div class="text-xs font-black line-clamp-2">${esc(p.name)}</div>
        <div class="mt-1 flex items-end gap-2">
          <div class="font-black brand-text">৳ ${esc(p.price)}</div>
          ${p.oldPrice>0 ? `<div class="text-xs text-gray-400 line-through">৳ ${esc(p.oldPrice)}</div>`:''}
        </div>
        <button class="mt-2 w-full py-2 r5 bg-slate-900 text-white text-xs font-black" data-add="${esc(p.id)}">Add to cart</button>
      </div>
    </div>
  `).join('');

  grid.querySelectorAll('[data-add]').forEach(b=>b.addEventListener('click', ()=>addToCart(b.getAttribute('data-add'))));
}

function addToCart(pid){
  state.cart[pid] = (state.cart[pid]||0) + 1;
  const cnt = Object.values(state.cart).reduce((a,b)=>a+b,0);
  const badge = document.getElementById('cartCount');
  if(badge) badge.textContent = String(cnt);
  renderCart();
}

async function boot(){
  try{
    state.settings = await api('/api/settings.php');
    setHeader();
    state.catalog = await api('/api/catalog.php');
    renderNavCategories();
    renderCatRows();
    const prod = await api('/api/products.php');
    state.slider = prod.sliderImages || [];
    state.products = prod.products || [];
    renderSlider();
    renderProducts();
    bindCartUI();
    syncCartCount();
    renderCart();
  }catch(e){
    console.error(e);
  }
}

document.addEventListener('DOMContentLoaded', boot);

function getProduct(pid){
  return state.products.find(p=>p.id===pid);
}

function renderCart(){
  const drawer = document.getElementById('cartDrawer');
  const itemsEl = document.getElementById('cartItems');
  const totalEl = document.getElementById('cartTotal');
  if(!itemsEl || !totalEl) return;

  const entries = Object.entries(state.cart).filter(([,q])=>q>0);
  let total = 0;
  itemsEl.innerHTML = entries.map(([pid,qty])=>{
    const p = getProduct(pid);
    const name = p? p.name : pid;
    const price = p? Number(p.price||0) : 0;
    total += price*qty;
    return `
      <div class="card p-3 flex items-center justify-between gap-2">
        <div>
          <div class="text-xs font-black">${esc(name)}</div>
          <div class="text-xs text-slate-500 font-bold">৳ ${price} × ${qty}</div>
        </div>
        <div class="flex items-center gap-2">
          <button class="px-3 py-2 border rounded-xl font-black text-xs" data-dec="${esc(pid)}">-</button>
          <button class="px-3 py-2 border rounded-xl font-black text-xs" data-inc="${esc(pid)}">+</button>
          <button class="px-3 py-2 border rounded-xl font-black text-xs" data-rem="${esc(pid)}">✕</button>
        </div>
      </div>
    `;
  }).join('') || '<div class="text-sm font-bold text-slate-500">Cart is empty</div>';

  totalEl.textContent = String(total.toFixed(0));

  itemsEl.querySelectorAll('[data-inc]').forEach(b=>b.onclick=()=>{ const id=b.getAttribute('data-inc'); state.cart[id]=(state.cart[id]||0)+1; syncCartCount(); renderCart(); });
  itemsEl.querySelectorAll('[data-dec]').forEach(b=>b.onclick=()=>{ const id=b.getAttribute('data-dec'); state.cart[id]=Math.max(1,(state.cart[id]||1)-1); syncCartCount(); renderCart(); });
  itemsEl.querySelectorAll('[data-rem]').forEach(b=>b.onclick=()=>{ const id=b.getAttribute('data-rem'); delete state.cart[id]; syncCartCount(); renderCart(); });
}

function syncCartCount(){
  const cnt = Object.values(state.cart).reduce((a,b)=>a+b,0);
  const badge = document.getElementById('cartCount');
  if(badge) badge.textContent = String(cnt);
}

function openCart(){
  const d=document.getElementById('cartDrawer');
  if(d){ d.classList.remove('hidden'); renderCart(); }
}
function closeCart(){
  const d=document.getElementById('cartDrawer');
  if(d) d.classList.add('hidden');
}

async function confirmOrder(){
  const name = document.getElementById('cName')?.value?.trim();
  const phone = document.getElementById('cPhone')?.value?.trim();
  const address = document.getElementById('cAddress')?.value?.trim();
  const msgEl = document.getElementById('orderMsg');
  const entries = Object.entries(state.cart).filter(([,q])=>q>0);
  if(!name || !phone || !entries.length){
    if(msgEl) msgEl.textContent='Please fill name/phone and add products.';
    return;
  }
  const payload = {
    name, phone, address,
    items: entries.map(([pid,qty])=>({id:pid, qty}))
  };
  try{
    if(msgEl) msgEl.textContent='Submitting order...';
    const res = await fetch('/api/order_create.php', {
      method:'POST',
      headers:{'Content-Type':'application/json'},
      cache:'no-store',
      body: JSON.stringify(payload)
    });
    const data = await res.json();
    if(!data.ok){
      if(msgEl) msgEl.textContent=data.error||'Order failed';
      return;
    }
    state.cart = {};
    syncCartCount();
    renderCart();
    if(msgEl) msgEl.innerHTML = `Order placed: <b>${esc(data.orderCode)}</b>. Invoice: <a class="brand-text underline" target="_blank" href="${esc(data.invoiceUrl)}">Open</a>`;
  }catch(e){
    if(msgEl) msgEl.textContent='Order failed';
  }
}

function bindCartUI(){
  document.getElementById('cartBtn')?.addEventListener('click', openCart);
  document.getElementById('cartClose')?.addEventListener('click', closeCart);
  document.getElementById('cartBackdrop')?.addEventListener('click', closeCart);
  document.getElementById('confirmOrderBtn')?.addEventListener('click', confirmOrder);
}
